## Generate a contour plot
# Import some other libraries that we'll need
# matplotlib and numpy packages must also be installed
import matplotlib
import numpy as np
import matplotlib.pyplot as plt

# define objective function
def f(x):
    w  = x[0]
    xc = x[1]
    obj = -(w-2.0*xc)*(0.8/w-2.0*xc)*xc
    return obj

# define objective gradient
def dfdx(x):
    w = x[0]
    xc = x[1]
    grad = []
    grad.append(-(1.6*xc**2 / w**2 - 2.0 * xc**2))
    grad.append(-(12.0*xc**2 - 3.2*xc/w - 4.0*w*xc + 0.8))
    return grad

# Exact 2nd derivatives (hessian)
def hes(x):
    w = x[0]
    xc = x[1]
    h = np.empty((2,2))
    h[0,0] = -(-3.2*xc**2 / w**3)
    h[0,1] = -(3.2*xc/w**2 - 4*xc)
    h[1,0] = h[0,1]
    h[1,1] = -(24.0*xc-4.0*w-3.2/w)
    return h

# Start location
x_start = [1.0, 0.1]

# Design variables at mesh points
i1 = np.arange(0, 1.8, 0.01)
i2 = np.arange(0, 0.65, 0.01)
width, xcut = np.meshgrid(i1, i2)
obj = (width-2.0*xcut)*(0.8/width-2.0*xcut)*xcut
# tab constraint
tabs = 5.0 / 100.0
box_width_with_tabs = width - 2.0*xcut + 2.0 * tabs
tab_con1 = width - box_width_with_tabs
tab_con2 = box_width_with_tabs - 2.0*tabs
x_con = box_width_with_tabs / 2.0 - xcut

# Create a contour plot
plt.figure()
# Specify contour lines
lines = np.linspace(0,0.052,10)
# Plot volume contours
CS = plt.contour(width, xcut, obj,lines)
# Label volume contours
plt.clabel(CS, inline=1, fontsize=10)
# Plot tab constraints
plt.contour(width, xcut, tab_con1,[0.0],colors='k',linewidths=[4.0])
plt.contour(width, xcut, tab_con2,[0.0],colors='b',linewidths=[4.0])
plt.contour(width, xcut, x_con,[0.0],colors='g',linewidths=[4.0])
plt.contour(width, xcut, tab_con1,[0.03,0.06],colors='k',linestyles='dashed',linewidths=[2.0,1.0])
plt.contour(width, xcut, tab_con2,[0.03,0.06],colors='b',linestyles='dashed',linewidths=[2.0,1.0])
plt.contour(width, xcut, x_con,[0.03,0.06],colors='g',linestyles='dashed',linewidths=[2.0,1.0])
# Add some text to the plot
plt.title('Optimizing a Box Volume')
plt.xlabel('cardboard width (m)')
plt.ylabel('box height (m)')
# Show the plot
#plt.show()


##################################################
# Newton's method
##################################################
# Number of iterations
n = 4
# Initialize xs
xn = np.zeros((n+1,2))
xn[0] = x_start
# Initialize delta_xn
delta_xn = np.empty((1,2))
# Get gradient at start location (df/dx or grad(f))
gn = np.zeros((n+1,1,2))
hn = np.zeros((n+1,2,2))

# Perform Newton iterations
for i in range(n):
    # calculate gradient
    gn[i] = dfdx(xn[i])
    
    # calculate hessian
    hn[i] = hes(xn[i])

    # Compute search direction and magnitude (dx)
    #  with dx = -inv(H) * grad
    delta_xn = -np.dot(np.linalg.pinv(hn[i]),gn[i].T)

    # Update next trial point
    xn[i+1] = xn[i] + delta_xn.T

# Plot Newton's iterations
plt.plot(xn[:,0],xn[:,1],'k-o')


##################################################
# Steepest descent method
##################################################
# Number of iterations
n = 20
# Use this alpha for the line search
alpha = np.linspace(0.2,0.2,n)
# Initialize xs
xs = np.zeros((n+1,2))
xs[0] = x_start
# Get gradient at start location (df/dx or grad(f))
for i in range(n):
    gs = dfdx(xs[i])
    # Compute search direction and magnitude (dx)
    #  with dx = - grad but no line searching
    xs[i+1] = xs[i] - np.dot(alpha[i],dfdx(xs[i]))
plt.plot(xs[:,0],xs[:,1],'g-o')

##################################################
# Conjugate gradient method
##################################################
# Number of iterations
n = 8
# Use this alpha for the line search
alpha = np.linspace(0.2,0.5,n)
neg = [[-1.0,0.0],[0.0,-1.0]]
# Initialize xc
xc = np.zeros((n+1,2))
xc[0] = x_start
# Initialize delta_gc
delta_cg = np.zeros((n+1,2))
# Initialize gc
gc = np.zeros((n+1,2))
# Get gradient at start location (df/dx or grad(f))
for i in range(n):
    gc[i] = dfdx(xc[i])
    # Compute search direction and magnitude (dx)
    #  with dx = - grad but no line searching
    if i==0:
        beta = 0
        delta_cg[i] = - np.dot(alpha[i],dfdx(xc[i]))
    else:
        beta = np.dot(gc[i],gc[i]) / np.dot(gc[i-1],gc[i-1])
        delta_cg[i] = alpha[i] * np.dot(neg,dfdx(xc[i])) + beta * delta_cg[i-1]
    xc[i+1] = xc[i] + delta_cg[i] 
plt.plot(xc[:,0],xc[:,1],'y-o')

##################################################
# Quasi-Newton method
##################################################
# Number of iterations
n = 8
# Use this alpha for every line search
alpha = np.linspace(0.2,1.0,n)
# Initialize delta_xq and gamma
delta_xq = np.zeros((2,1))
gamma = np.zeros((2,1))
part1 = np.zeros((2,2))
part2 = np.zeros((2,2))
part3 = np.zeros((2,2))
part4 = np.zeros((2,2))
part5 = np.zeros((2,2))
part6 = np.zeros((2,1))
part7 = np.zeros((1,1))
part8 = np.zeros((2,2))
part9 = np.zeros((2,2))
# Initialize xq
xq = np.zeros((n+1,2))
xq[0] = x_start
# Initialize gradient storage
g = np.zeros((n+1,2))
g[0] = dfdx(xq[0])
# Initialize hessian storage
h = np.zeros((n+1,2,2))
h[0] = [[1.0, 0.0],[0.0, 1.0]]
for i in range(n):
    # Compute search direction and magnitude (dx)
    #  with dx = -alpha * inv(h) * grad
    delta_xq = -np.dot(alpha[i],np.linalg.solve(h[i],g[i]))
    xq[i+1] = xq[i] + delta_xq

    # Get gradient update for next step
    g[i+1] = dfdx(xq[i+1])

    # Get hessian update for next step
    gamma = g[i+1]-g[i]
    part1 = np.outer(gamma,gamma)
    part2 = np.outer(gamma,delta_xq)
    part3 = np.dot(np.linalg.pinv(part2),part1)

    part4 = np.outer(delta_xq,delta_xq)
    part5 = np.dot(h[i],part4)
    part6 = np.dot(part5,h[i])
    part7 = np.dot(delta_xq,h[i])
    part8 = np.dot(part7,delta_xq)
    part9 = np.dot(part6,1/part8)
    
    h[i+1] = h[i] + part3 - part9

plt.plot(xq[:,0],xq[:,1],'r-o')

# Save the figure as a PNG
plt.savefig('contour.png')

plt.show()
