close all
clear all

addpath('apm');

% -------------------------------------------------------------------------------
% Step response to generate data
% -------------------------------------------------------------------------------

% define server and application
s = 'http://xps.apmonitor.com';
a = 'est';

% clear previous application
apm(s,a,'clear all');

% load model
apm_load(s,a,'step.apm');

% specify parameters or variables of interest
apm_info(s,a,'FV','K');
apm_info(s,a,'MV','u');
apm_info(s,a,'CV','x');

% load step test
csv_load(s,a,'step.csv');

% solve sequentially
apm_option(s,a,'nlc.imode',7);

% turn off historical trending
apm_option(s,a,'nlc.hist_hor',0);

% solve
apm(s,a,'solve');

% retrieve solution file
step = apm_sol(s,a);

% -------------------------------------------------------------------------------
% Parameter Estimation
% -------------------------------------------------------------------------------

% clear previous csv file
apm(s,a,'clear csv');

% load new csv file
csv_load(s,a,'solution.csv');

% switch to parameter estimation mode
apm_option(s,a,'nlc.imode',5);

% estimate K (gain)
apm_option(s,a,'k.status',1);

% turn on feedback status for variables to estimate
apm_option(s,a,'x.fstatus',1);

% solve parameter estimation
apm(s,a,'solve');

% retrieve value of K (gain)
k_val = apm_tag(s,a,'k.newval');
disp('Actual Parameter: 3.5');
disp(['Estimated Parameter: ' num2str(k_val)]);

% retrieve results
results = apm_sol(s,a);

% -------------------------------------------------------------------------------
% Trending
% -------------------------------------------------------------------------------

% parse values for trending
names = results(1,:);
vals = cell2mat(results(2:end,:));
[m,n] = size(vals);
% time is the first column
t = vals(:,1);
% retrieve column indices for values of interest
iu = find(strcmpi('u',names));
ix = find(strcmpi('x',names));
ik = find(strcmpi('k',names));
u = vals(:,iu);
x = vals(:,ix);
k = vals(:,ik);

% parse step test results for trending
names = step(1,:);
vals = cell2mat(step(2:end,:));
[m,n] = size(vals);
% time is the first column
ts = vals(:,1);
% retrieve column indices for values of interest
iu = find(strcmpi('u',names));
ix = find(strcmpi('x',names));
ik = find(strcmpi('k',names));
us = vals(:,iu);
xs = vals(:,ix);
ks = vals(:,ik);

% plot results
figure(1)
subplot(2,1,1)
plot(ts,us,'k.');
hold on
plot(ts,xs,'k-.');
plot(t,x,'b-');
legend('u_{data}','x_{data}','x_{est}');
subplot(2,1,2)
plot(ts,ks,'k-');
hold on
plot(t,k,'r-');
legend('k_{data}','k_{est}');

% open web viewer
apm_web(s,a);

% -------------------------------------------------------------------------------
% Optimization under uncertainty
% -------------------------------------------------------------------------------

% change application name
a = 'opt';

% clear previous application
apm(s,a,'clear all');

% load model
apm_load(s,a,'ctl.apm');

% number of sample points
n = 10;

% specify parameters or variables of interest
apm_info(s,a,'MV','u');
for i=1:n,
  k_name = ['K[' int2str(i) ']'];
  apm_info(s,a,'FV',k_name);
  x_name = ['x[' int2str(i) ']'];
  apm_info(s,a,'CV',x_name);
end

% load step test
csv_load(s,a,'ctl.csv');

% get standard deviation for K (gain)
k_stdev = std(ks-k);
k_mean = k_val;

% generate sample points between 3 and 4
k_samp = linspace(3,4,n);

% get probability of sample points - will become weighting on CVs
k_pdf = pdf('norm',k_samp,k_mean,k_stdev);

% modify the gain for the simulation
for i=1:n,
  k_name = ['K[' int2str(i) ']'];
  apm_meas(s,a,k_name,k_samp(i));
end

% manipulated variable tuning
apm_option(s,a,'u.status',1);

% controlled variable tuning
sphi = 10.2;
splo = 9.8;
for i=1:n,
  x_name = ['x[' int2str(i) ']'];
  apm_option(s,a,[x_name '.status'],1);
  apm_option(s,a,[x_name '.tau'],3);
  apm_option(s,a,[x_name '.sphi'],sphi);
  apm_option(s,a,[x_name '.splo'],splo);
  apm_option(s,a,[x_name '.wsphi'],k_pdf(i)*100);
  apm_option(s,a,[x_name '.wsplo'],k_pdf(i)*100);
end

% solve in control mode
apm_option(s,a,'nlc.imode',6);

% solve in control mode
apm_option(s,a,'nlc.reqctrlmode',3);

% turn off historical trending
apm_option(s,a,'nlc.hist_hor',0);

% solve
solve = apm(s,a,'solve');

% retrieve results
results = apm_sol(s,a);

% -------------------------------------------------------------------------------
% Trending
% -------------------------------------------------------------------------------

% parse values for trending
heading = results(1,:);
v = cell2mat(results(2:end,:));
% time is the first column
t_ctl = v(:,1);
% retrieve column indices for values of interest
iu = find(strcmpi('u',heading));
u_ctl = v(:,iu);
% retrieve x[i]
ix = find(strcmpi('x[1]',heading));
x_ctl = v(:,ix:ix+n-1);

% plot results
figure(2)
subplot(2,1,1)
plot(t_ctl,u_ctl,'k-');
ylabel('Manipulated');
legend('u_{opt}');
subplot(2,1,2)
for i = 1:n,
  plot(t_ctl,x_ctl(:,i),'k-');
  hold on
end
ylabel('Controlled');
legend('x_{opt}');

% open web viewer
apm_web(s,a);
